unit DebugTFDD;

interface

{$REGION 'macOS stuff'}
{$IFDEF MACOS}
type
  TInternalMacOSLogger = class
  public
    class procedure Log(const Msg: string);
  end;
{$ENDIF}
{$ENDREGION}

type
  TDebugRoutine = procedure(const S: String) of object;

var
  DebugRoutine: TDebugRoutine = nil;

implementation

uses
{$IFDEF POSIX}
  Posix.Stdlib, // for setenv
{$ENDIF}
{$IFDEF MACOS}
  Macapi.Helpers, // for StringToID
  Macapi.CoreFoundation, // for _PU
{$ENDIF}
{$IFDEF MSWINDOWS}
  Winapi.Windows, // for SetEnvironmentVariable
{$ENDIF}
  System.SysUtils; // for TMarshaller, GetEnvironmentVariable

{$REGION 'macOS stuff'}
{$IFDEF MACOS}
const
  libFoundation = '/System/Library/Frameworks/Foundation.framework/Foundation';

type
  PNSString = Pointer;

procedure NSLog(format: PNSString); cdecl; varargs; external libFoundation name _PU + 'NSLog';

class procedure TInternalMacOSLogger.Log(const Msg: string);
begin
  NSLog(StringToID(Msg));
end;
{$ENDIF}
{$ENDREGION}

function DebugOpen(var F: TTextRec): Integer; forward;
function DebugOutput(var F: TTextRec): Integer; forward;
function DebugClose(var F: TTextRec): Integer; forward;

procedure AssignDebug(var F: TextFile);
begin
  { Set up text file variable }
  TTextRec(F).Handle := $FFFF;
  TTextRec(F).OpenFunc := @DebugOpen;
  TTextRec(F).Mode := fmClosed;
  TTextRec(F).BufSize := SizeOf(TTextRec(F).Buffer);
  TTextRec(F).BufPtr := @TTextRec(F).Buffer;
  TTextRec(F).Name[0] := #0;
end;

function DebugOpen(var F: TTextRec): Integer;
begin
  Result := 0;
  if F.Mode = fmInput then
  begin
    Result := 5 // Access denied
  end
  else
  begin
    F.Mode := fmOutput;
    F.InOutFunc := @DebugOutput;
    F.FlushFunc := @DebugOutput;
  end;
  F.CloseFunc := @DebugClose;
end;

var
  DebugMsg: String = '';

function DebugOutput(var F: TTextRec): Integer;
var
  // RawByteString is conveniently available on mobile platforms, unlike AnsiString and ShortString
  Txt: RawByteString;
begin
  Result := 0;
  if F.BufPos > 0 then
  begin
    SetLength(Txt, F.BufPos);
    SetCodePage(Txt, {$IFDEF NEXTGEN}CP_UTF8{$ELSE}DefaultSystemCodePage{$ENDIF});
    Move(F.BufPtr^, Txt[{$IFDEF NEXTGEN}0{$ELSE}1{$ENDIF}], F.BufPos);
    F.BufPos := 0;
    if Txt = #$D then
    begin
      // Just swallow any lone carriage return characters
    end
    else
    begin
      DebugMsg := DebugMsg + String(Txt);
    end;
    if Txt = #$A then
    begin
      // We can't directly call Log.d here, as this requires FMX.Types in
      // the uses clause, which will make FMX.Types get
      // initialised too early in the unit initialisation list.
      // This causes a crash on exit due to finalisation order..... :/
      // So we'll have to do it indirectly via a pluggable pointer,
      // which in this case is plugged in by the project source code
      if Assigned(DebugRoutine) then
        DebugRoutine(DebugMsg);
      DebugMsg := '';
    end;
  end;
end;

function DebugClose(var F: TTextRec): Integer;
begin
  Result := 0;
end;

{$REGION 'TDebugUtils.DebugPrint stuff'}
procedure SetEnvironmentVariable(const Name, Value: string);
{$IFDEF MSWINDOWS}
begin
  Winapi.Windows.SetEnvironmentVariable(PChar(Name), PChar(Value))
end;
{$ENDIF}
{$IFDEF POSIX}
var
  M: TMarshaller;
begin
  setenv(M.AsAnsi(Name, CP_UTF8).ToPointer, M.AsAnsi(Value, CP_UTF8).ToPointer, 1);
end;
{$ENDIF}
{$ENDREGION}

initialization
{$REGION 'TDebugUtils.DebugPrint stuff'}
  // The class constructor runs during app startup & parses the DEBUG_CLASS
  // environment variable. Because of this we must set the env var value
  // (if doing so in the program) up front in the initialisation section of
  // a unit that is initialised before System.Internal.DebugUtils.
  // This is why DebugTFDD must be first in the project file uses clause.
  // If we are setting up DEBUG_CLASS in the environment outside the program,
  // then this is not a requirement.
  if GetEnvironmentVariable('DEBUG_CLASS') = '' then
    SetEnvironmentVariable('DEBUG_CLASS', '*');
{$ENDREGION}

  // If you are working with a console application, then you may
  // prefer to pass a different TextFile variable into AssignDebug
  AssignDebug(Output);
  Rewrite(Output);
end.
